import React, { useState, useEffect, useRef } from 'react';
import { useNavigate, useParams } from 'react-router-dom';
import {
  Box,
  Button,
  Card,
  CardContent,
  Chip,
  Container,
  Dialog,
  DialogActions,
  DialogContent,
  DialogContentText,
  DialogTitle,
  Divider,
  FormControl,
  Grid,
  InputLabel,
  LinearProgress,
  MenuItem,
  Paper,
  Select,
  Stack,
  Step,
  StepLabel,
  Stepper,
  Switch,
  TextField,
  Typography,
  FormControlLabel,
} from '@mui/material';
import PlayArrowIcon from '@mui/icons-material/PlayArrow';
import CodeIcon from '@mui/icons-material/Code';
import CloudIcon from '@mui/icons-material/Cloud';
import SettingsIcon from '@mui/icons-material/Settings';
import BugReportIcon from '@mui/icons-material/BugReport';
import CheckCircleOutlineIcon from '@mui/icons-material/CheckCircleOutline';
import AccessTimeIcon from '@mui/icons-material/AccessTime';

const testSteps = ['Конфигурация', 'Тестирование', 'Результаты'];

const cloudOptions = [
  { value: 'yandex', label: 'Яндекс Облако' },
  { value: 'sber', label: 'SberCloud' },
  { value: 'mail', label: 'Mail.ru Cloud Solutions' },
  { value: 'selectel', label: 'Selectel' },
];

const scenarioOptions = [
  { value: 'api', label: 'API Gateway' },
  { value: 'processing', label: 'Обработка данных' },
  { value: 'database', label: 'Работа с БД' },
  { value: 'images', label: 'Обработка изображений' },
  { value: 'custom', label: 'Пользовательский сценарий' },
];

// Моковые данные для выбора функции
const mockFunctions = [
  {
    id: 'func1',
    name: 'API Gateway Handler',
    versions: [
      { version: 1, label: 'Версия 1' },
      { version: 2, label: 'Версия 2' },
      { version: 3, label: 'Версия 3' }
    ]
  },
  {
    id: 'func2',
    name: 'Database Worker',
    versions: [
      { version: 1, label: 'Версия 1' },
      { version: 2, label: 'Версия 2' }
    ]
  },
  {
    id: 'func3',
    name: 'Image Processor',
    versions: [
      { version: 1, label: 'Версия 1' }
    ]
  }
];

// Моковые данные для рантаймов
const runtimeOptions = [
  { value: 'python310', label: 'Python 3.10' },
  { value: 'python39', label: 'Python 3.9' },
  { value: 'nodejs16', label: 'Node.js 16' },
  { value: 'nodejs14', label: 'Node.js 14' },
  { value: 'go119', label: 'Go 1.19' }
];

function TestingPage() {
  const navigate = useNavigate();
  const params = useParams();
  const { functionId, version } = params;
  
  // Состояния
  const [activeStep, setActiveStep] = useState(0);
  const [iterations, setIterations] = useState(100);
  const [parallelRuns, setParallelRuns] = useState(10);
  const [timeout, setTimeout] = useState(5000);
  const [progress, setProgress] = useState(0);
  const [testStartTime, setTestStartTime] = useState(null);
  const [testDuration, setTestDuration] = useState(null);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [selectedFunction, setSelectedFunction] = useState(functionId || '');
  const [selectedVersion, setSelectedVersion] = useState(version ? parseInt(version) : '');
  const [selectedRuntime, setSelectedRuntime] = useState('');
  const [versions, setVersions] = useState([]);
  
  // Для таймера
  const timerRef = useRef(null);
  const [elapsedTime, setElapsedTime] = useState(0);
  
  // Проверка валидности формы
  const isFormValid = selectedFunction && selectedVersion && selectedRuntime && 
                      iterations > 0 && parallelRuns > 0 && timeout > 0;
  
  // Эффект для инициализации выбранной функции и версии из URL
  useEffect(() => {
    if (functionId) {
      const func = mockFunctions.find(f => f.id === functionId);
      if (func) {
        setSelectedFunction(functionId);
        setVersions(func.versions);
        
        if (version) {
          const versionInt = parseInt(version);
          const versionExists = func.versions.some(v => v.version === versionInt);
          if (versionExists) {
            setSelectedVersion(versionInt);
          }
        }
      }
    }
  }, [functionId, version]);
  
  // Эффект для обновления доступных версий при выборе функции
  useEffect(() => {
    if (selectedFunction) {
      const func = mockFunctions.find(f => f.id === selectedFunction);
      if (func) {
        setVersions(func.versions);
        if (func.versions.length > 0 && !selectedVersion) {
          setSelectedVersion(func.versions[0].version);
        }
      }
    } else {
      setVersions([]);
      setSelectedVersion('');
    }
  }, [selectedFunction]);
  
  // Эффект для таймера
  useEffect(() => {
    if (activeStep === 1 && testStartTime) {
      timerRef.current = setInterval(() => {
        const elapsed = Math.floor((Date.now() - testStartTime) / 1000);
        setElapsedTime(elapsed);
      }, 1000);
      
      return () => clearInterval(timerRef.current);
    }
  }, [activeStep, testStartTime]);
  
  // Эффект для симуляции прогресса тестирования
  useEffect(() => {
    if (activeStep === 1) {
      const interval = setInterval(() => {
        setProgress(prevProgress => {
          const newProgress = prevProgress + 1;
          if (newProgress >= 100) {
            clearInterval(interval);
            // Тестирование завершено
            const duration = Math.floor((Date.now() - testStartTime) / 1000);
            setTestDuration(duration);
            setActiveStep(2);
            clearInterval(timerRef.current);
            return 100;
          }
          return newProgress;
        });
      }, 150);
      
      return () => clearInterval(interval);
    }
  }, [activeStep, testStartTime]);
  
  // Обработчики
  const handleStartTest = () => {
    setActiveStep(1);
    setProgress(0);
    setTestStartTime(Date.now());
    setElapsedTime(0);
  };
  
  const handleReset = () => {
    setActiveStep(0);
    setProgress(0);
  };
  
  const handleFunctionChange = (event) => {
    setSelectedFunction(event.target.value);
  };
  
  const handleVersionChange = (event) => {
    setSelectedVersion(event.target.value);
  };
  
  const handleRuntimeChange = (event) => {
    setSelectedRuntime(event.target.value);
  };
  
  const handleCancelTest = () => {
    setIsDialogOpen(true);
  };
  
  const handleConfirmCancel = () => {
    clearInterval(timerRef.current);
    setIsDialogOpen(false);
    setActiveStep(0);
    setProgress(0);
    setTestStartTime(null);
  };
  
  const handleDialogClose = () => {
    setIsDialogOpen(false);
  };
  
  const handleViewResults = () => {
    navigate(`/results?testId=${selectedFunction}-${selectedVersion}`);
  };
  
  const handleNewTest = () => {
    setActiveStep(0);
    setProgress(0);
    setTestStartTime(null);
    setTestDuration(null);
  };
  
  // Форматирование времени
  const formatTime = (seconds) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
  };
  
  // Получение названия выбранной функции и версии
  const getSelectedFunctionName = () => {
    if (selectedFunction && selectedVersion) {
      const func = mockFunctions.find(f => f.id === selectedFunction);
      if (func) {
        return `${func.name} v${selectedVersion}`;
      }
    }
    return '';
  };

  return (
    <Container maxWidth="lg">
      <Box sx={{ my: 4 }}>
        <Typography variant="h4" component="h1" gutterBottom>
          Тестирование Serverless-функций
        </Typography>
        <Typography variant="body1" paragraph>
          Настройте параметры и запустите тестирование функций на выбранной облачной платформе.
        </Typography>

        <Paper sx={{ p: 3, mb: 4 }}>
          <Stepper activeStep={activeStep} sx={{ mb: 4 }}>
            {testSteps.map((label) => (
              <Step key={label}>
                <StepLabel>{label}</StepLabel>
              </Step>
            ))}
          </Stepper>

          {activeStep === 0 && (
            <Box>
              <Typography variant="h6" gutterBottom>
                Настройки тестирования
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={6}>
                  <FormControl fullWidth margin="normal">
                    <InputLabel>Функция</InputLabel>
                    <Select
                      value={selectedFunction}
                      label="Функция"
                      onChange={handleFunctionChange}
                    >
                      {mockFunctions.map((func) => (
                        <MenuItem key={func.id} value={func.id}>
                          {func.name}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                  
                  <FormControl fullWidth margin="normal" disabled={!selectedFunction}>
                    <InputLabel>Версия</InputLabel>
                    <Select
                      value={selectedVersion}
                      label="Версия"
                      onChange={handleVersionChange}
                    >
                      {versions.map((ver) => (
                        <MenuItem key={ver.version} value={ver.version}>
                          {ver.label}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                  
                  <FormControl fullWidth margin="normal">
                    <InputLabel>Рантайм</InputLabel>
                    <Select
                      value={selectedRuntime}
                      label="Рантайм"
                      onChange={handleRuntimeChange}
                    >
                      {runtimeOptions.map((option) => (
                        <MenuItem key={option.value} value={option.value}>
                          {option.label}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                </Grid>
                
                <Grid item xs={12} md={6}>
                  <TextField
                    fullWidth
                    margin="normal"
                    label="Количество итераций"
                    type="number"
                    value={iterations}
                    onChange={(e) => setIterations(Number(e.target.value))}
                    inputProps={{ min: 1 }}
                  />
                  
                  <TextField
                    fullWidth
                    margin="normal"
                    label="Параллельные запуски"
                    type="number"
                    value={parallelRuns}
                    onChange={(e) => setParallelRuns(Number(e.target.value))}
                    inputProps={{ min: 1 }}
                  />
                  
                  <TextField
                    fullWidth
                    margin="normal"
                    label="Таймаут (мс)"
                    type="number"
                    value={timeout}
                    onChange={(e) => setTimeout(Number(e.target.value))}
                    inputProps={{ min: 100 }}
                  />
                </Grid>
              </Grid>
              
              <Box sx={{ mt: 4, display: 'flex', justifyContent: 'flex-end' }}>
                <Button
                  variant="contained"
                  color="primary"
                  onClick={handleStartTest}
                  disabled={!isFormValid}
                >
                  Запустить тестирование
                </Button>
              </Box>
            </Box>
          )}

          {activeStep === 1 && (
            <Box>
              <Typography variant="h6" gutterBottom>
                Выполнение тестирования
              </Typography>
              
              <Card sx={{ mb: 4, bgcolor: '#f8f8f8' }}>
                <CardContent>
                  <Typography variant="subtitle1" gutterBottom fontWeight="bold">
                    Выбранные параметры:
                  </Typography>
                  <Grid container spacing={2}>
                    <Grid item xs={12} sm={6}>
                      <Typography variant="body2">
                        <strong>Функция:</strong> {getSelectedFunctionName()}
                      </Typography>
                      <Typography variant="body2">
                        <strong>Рантайм:</strong> {runtimeOptions.find(r => r.value === selectedRuntime)?.label || ''}
                      </Typography>
                    </Grid>
                    <Grid item xs={12} sm={6}>
                      <Typography variant="body2">
                        <strong>Итераций:</strong> {iterations}
                      </Typography>
                      <Typography variant="body2">
                        <strong>Параллельных запусков:</strong> {parallelRuns}
                      </Typography>
                    </Grid>
                  </Grid>
                </CardContent>
              </Card>
              
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }}>
                <AccessTimeIcon color="primary" sx={{ mr: 1 }} />
                <Typography variant="h6" component="div">
                  {formatTime(elapsedTime)}
                </Typography>
              </Box>
              
              <Typography variant="body2" gutterBottom>
                Прогресс тестирования:
              </Typography>
              <LinearProgress 
                variant="determinate" 
                value={progress} 
                sx={{ height: 10, borderRadius: 5, mb: 1 }} 
              />
              <Typography variant="body2" color="text.secondary" sx={{ mb: 4 }}>
                {Math.floor(iterations * progress / 100)} из {iterations} итераций ({progress}%)
              </Typography>
              
              <Box sx={{ display: 'flex', justifyContent: 'center' }}>
                <Button
                  variant="outlined"
                  color="error"
                  onClick={handleCancelTest}
                >
                  Отмена
                </Button>
              </Box>
            </Box>
          )}

          {activeStep === 2 && (
            <Box sx={{ display: 'flex', flexDirection: 'column', alignItems: 'center', py: 4 }}>
              <CheckCircleOutlineIcon color="success" sx={{ fontSize: 80, mb: 2 }} />
              
              <Typography variant="h6" gutterBottom align="center">
                Тестирование функции "{getSelectedFunctionName()}" завершено за {formatTime(testDuration)}.
              </Typography>
              
              <Typography variant="body1" align="center" sx={{ mb: 4 }}>
                Результаты тестирования сохранены.
              </Typography>
              
              <Stack direction="row" spacing={2}>
                <Button
                  variant="outlined"
                  onClick={handleNewTest}
                >
                  Новый тест
                </Button>
                <Button
                  variant="contained"
                  onClick={handleViewResults}
                >
                  Просмотр результатов
                </Button>
              </Stack>
            </Box>
          )}
        </Paper>
        
        {/* Диалог подтверждения отмены */}
        <Dialog
          open={isDialogOpen}
          onClose={handleDialogClose}
        >
          <DialogTitle>Подтверждение отмены</DialogTitle>
          <DialogContent>
            <DialogContentText>
              Вы уверены, что хотите отменить текущее тестирование? 
              Все промежуточные результаты будут потеряны.
            </DialogContentText>
          </DialogContent>
          <DialogActions>
            <Button onClick={handleDialogClose}>Нет</Button>
            <Button onClick={handleConfirmCancel} color="error" autoFocus>
              Да, отменить
            </Button>
          </DialogActions>
        </Dialog>
      </Box>
    </Container>
  );
}

export default TestingPage; 