import React, { useState, useEffect } from 'react';
import {
  Alert,
  AlertTitle,
  Box,
  Button,
  Card,
  CardContent,
  CircularProgress,
  Container,
  FormControl,
  Grid,
  InputLabel,
  ListSubheader,
  MenuItem,
  Paper,
  Select,
  Tab,
  Tabs,
  Typography,
  useTheme,
} from '@mui/material';
import CodeIcon from '@mui/icons-material/Code';
import JavascriptIcon from '@mui/icons-material/Javascript';
import { Line } from 'react-chartjs-2';
import {
  Chart as ChartJS,
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  Title,
  Tooltip as ChartTooltip,
  Legend,
} from 'chart.js';

// Регистрация компонентов ChartJS
ChartJS.register(
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  Title,
  ChartTooltip,
  Legend
);

// Моковые данные для тестовых профилей
const mockProfiles = [
  {
    functionId: 'func1',
    functionName: 'API Gateway Handler',
    language: 'python',
    versions: [
      {
        versionId: 1,
        versionLabel: 'Версия 1',
        tests: [
          {
            id: 'test1',
            date: '2024-06-14 10:30',
            runtime: 'python310',
            platform: 'Яндекс Облако',
            iterations: 100,
            concurrency: 10,
            metrics: {
              coldStart: {
                p25: 250,
                p50: 280,
                p75: 310,
                p100: 350
              },
              executionTime: {
                p25: 120,
                p50: 135,
                p75: 150,
                p100: 180
              },
              cpu: {
                p25: 20,
                p50: 25,
                p75: 32,
                p100: 45
              },
              memory: {
                p25: 180,
                p50: 230,
                p75: 260,
                p100: 300
              },
              network: {
                p25: 15,
                p50: 22,
                p75: 30,
                p100: 45
              }
            }
          },
          {
            id: 'test2',
            date: '2024-06-15 14:45',
            runtime: 'python310',
            platform: 'Яндекс Облако',
            iterations: 150,
            concurrency: 15,
            metrics: {
              coldStart: {
                p25: 245,
                p50: 275,
                p75: 305,
                p100: 345
              },
              executionTime: {
                p25: 115,
                p50: 130,
                p75: 145,
                p100: 175
              },
              cpu: {
                p25: 18,
                p50: 23,
                p75: 30,
                p100: 42
              },
              memory: {
                p25: 175,
                p50: 225,
                p75: 255,
                p100: 295
              },
              network: {
                p25: 14,
                p50: 20,
                p75: 27,
                p100: 42
              }
            }
          }
        ]
      },
      {
        versionId: 2,
        versionLabel: 'Версия 2',
        tests: [
          {
            id: 'test3',
            date: '2024-06-16 09:15',
            runtime: 'python310',
            platform: 'Яндекс Облако',
            iterations: 100,
            concurrency: 10,
            metrics: {
              coldStart: {
                p25: 220,
                p50: 245,
                p75: 275,
                p100: 320
              },
              executionTime: {
                p25: 105,
                p50: 120,
                p75: 135,
                p100: 165
              },
              cpu: {
                p25: 15,
                p50: 20,
                p75: 27,
                p100: 38
              },
              memory: {
                p25: 160,
                p50: 200,
                p75: 235,
                p100: 275
              },
              network: {
                p25: 12,
                p50: 18,
                p75: 25,
                p100: 38
              }
            }
          }
        ]
      }
    ]
  },
  {
    functionId: 'func2',
    functionName: 'Database Worker',
    language: 'nodejs',
    versions: [
      {
        versionId: 1,
        versionLabel: 'Версия 1',
        tests: [
          {
            id: 'test4',
            date: '2024-06-14 15:30',
            runtime: 'nodejs16',
            platform: 'Яндекс Облако',
            iterations: 100,
            concurrency: 8,
            metrics: {
              coldStart: {
                p25: 260,
                p50: 290,
                p75: 320,
                p100: 360
              },
              executionTime: {
                p25: 130,
                p50: 145,
                p75: 160,
                p100: 190
              },
              cpu: {
                p25: 22,
                p50: 28,
                p75: 35,
                p100: 50
              },
              memory: {
                p25: 195,
                p50: 245,
                p75: 275,
                p100: 320
              },
              network: {
                p25: 18,
                p50: 25,
                p75: 33,
                p100: 48
              }
            }
          }
        ]
      }
    ]
  }
];

// Состояния страницы
const PAGE_STATES = {
  SELECTION: 'selection',
  VIEW: 'view',
  COMPARISON: 'comparison',
};

// Метрики для отображения
const METRICS = [
  { id: 'coldStart', label: 'Время холодного старта (мс)' },
  { id: 'executionTime', label: 'Время выполнения (мс)' },
  { id: 'cpu', label: 'Потребление ЦП (%)' },
  { id: 'memory', label: 'Потребление памяти (МБ)' },
  { id: 'network', label: 'Потребление сетевого трафика (КБ/с)' },
];

// Общие опции для графиков
const getChartOptions = (title) => ({
  responsive: true,
  maintainAspectRatio: false,
  plugins: {
    legend: {
      position: 'top',
    },
    title: {
      display: true,
      text: title,
    },
  },
});

function ResultsPage() {
  const theme = useTheme();
  const [pageState, setPageState] = useState(PAGE_STATES.SELECTION);
  const [profiles, setProfiles] = useState(mockProfiles);
  const [selectedProfile1, setSelectedProfile1] = useState('');
  const [selectedProfile2, setSelectedProfile2] = useState('');
  const [currentTabIndex, setCurrentTabIndex] = useState(0);
  const [profileData1, setProfileData1] = useState(null);
  const [profileData2, setProfileData2] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [compatibleProfiles, setCompatibleProfiles] = useState([]);

  // Обработка изменения вкладки метрики
  const handleTabChange = (event, newValue) => {
    setCurrentTabIndex(newValue);
  };

  // Обработка выбора первого профиля
  const handleProfile1Change = (event) => {
    const profile = event.target.value;
    setSelectedProfile1(profile);
    setSelectedProfile2('');
    
    if (profile) {
      // Поиск данных профиля
      const [funcId, verId, testId] = profile.split(':');
      
      // Поиск совместимых профилей (той же функции, но более новых версий)
      const selectedFunction = profiles.find(f => f.functionId === funcId);
      if (selectedFunction) {
        const selectedVersionIndex = selectedFunction.versions.findIndex(v => v.versionId.toString() === verId);
        
        if (selectedVersionIndex !== -1) {
          // Собираем все профили тестов более новых версий
          const compatibleOptions = [];
          
          for (let i = selectedVersionIndex + 1; i < selectedFunction.versions.length; i++) {
            const version = selectedFunction.versions[i];
            version.tests.forEach(test => {
              compatibleOptions.push({
                id: `${funcId}:${version.versionId}:${test.id}`,
                functionName: selectedFunction.functionName,
                versionLabel: version.versionLabel,
                testDate: test.date
              });
            });
          }
          
          setCompatibleProfiles(compatibleOptions);
        }
      }
      
      // Загрузка данных профиля
      loadProfileData(profile, setProfileData1);
      setPageState(PAGE_STATES.VIEW);
    } else {
      setProfileData1(null);
      setPageState(PAGE_STATES.SELECTION);
    }
  };

  // Обработка выбора второго профиля
  const handleProfile2Change = (event) => {
    const profile = event.target.value;
    setSelectedProfile2(profile);
    
    if (profile) {
      // Загрузка данных профиля
      loadProfileData(profile, setProfileData2);
      setPageState(PAGE_STATES.COMPARISON);
    } else {
      setProfileData2(null);
      setPageState(PAGE_STATES.VIEW);
    }
  };

  // Загрузка данных профиля
  const loadProfileData = (profileId, setData) => {
    if (!profileId) return;
    
    setLoading(true);
    setError(null);
    
    try {
      // Парсинг ID профиля
      const [funcId, versionId, testId] = profileId.split(':');
      
      // Поиск данных в моковом наборе
      const selectedFunction = profiles.find(f => f.functionId === funcId);
      if (selectedFunction) {
        const selectedVersion = selectedFunction.versions.find(v => v.versionId.toString() === versionId);
        if (selectedVersion) {
          const selectedTest = selectedVersion.tests.find(t => t.id === testId);
          if (selectedTest) {
            const profileData = {
              functionId: funcId,
              functionName: selectedFunction.functionName,
              language: selectedFunction.language,
              versionId: versionId,
              versionLabel: selectedVersion.versionLabel,
              ...selectedTest
            };
            
            // Имитация асинхронной загрузки
            setTimeout(() => {
              setData(profileData);
              setLoading(false);
            }, 500);
            return;
          }
        }
      }
      
      throw new Error('Профиль не найден');
    } catch (err) {
      console.error('Error loading profile:', err);
      setError('Не удалось загрузить данные профиля');
      setLoading(false);
    }
  };

  // Получение данных для метрик
  const getMetricChartData = (metric) => {
    const isComparisonMode = pageState === PAGE_STATES.COMPARISON && profileData2;
    
    // Проверка наличия данных для метрики
    if (!profileData1?.metrics[metric]) {
      return null;
    }
    
    const labels = ['p25', 'p50', 'p75', 'p100'];
    const datasets = [
      {
        label: `${profileData1.functionName} ${profileData1.versionLabel}`,
        data: [
          profileData1.metrics[metric].p25,
          profileData1.metrics[metric].p50,
          profileData1.metrics[metric].p75,
          profileData1.metrics[metric].p100
        ],
        borderColor: theme.palette.primary.main,
        backgroundColor: `${theme.palette.primary.main}20`,
        tension: 0.3,
      }
    ];
    
    if (isComparisonMode && profileData2?.metrics[metric]) {
      datasets.push({
        label: `${profileData2.functionName} ${profileData2.versionLabel}`,
        data: [
          profileData2.metrics[metric].p25,
          profileData2.metrics[metric].p50,
          profileData2.metrics[metric].p75,
          profileData2.metrics[metric].p100
        ],
        borderColor: theme.palette.secondary.main,
        backgroundColor: `${theme.palette.secondary.main}20`,
        tension: 0.3,
      });
    }
    
    return {
      labels: labels.map(p => p.toUpperCase().replace('P', '')),
      datasets
    };
  };

  // Построение древовидного списка профилей для селектора
  const renderProfilesMenu = () => {
    if (profiles.length === 0) {
      return [
        <MenuItem key="empty" disabled>
          Нет сохранённых результатов
        </MenuItem>
      ];
    }

    return [
      <MenuItem key="default" value="" divider>
        Выберите профиль
      </MenuItem>,
      ...profiles.map(func => [
        <ListSubheader key={`func-${func.functionId}`} sx={{ display: 'flex', alignItems: 'center' }}>
          {func.language === 'python' ? 
            <CodeIcon fontSize="small" color="primary" sx={{ mr: 1 }} /> : 
            <JavascriptIcon fontSize="small" color="warning" sx={{ mr: 1 }} />}
          {func.functionName}
        </ListSubheader>,
        ...func.versions.flatMap(ver => [
          <MenuItem 
            key={`ver-${func.functionId}-${ver.versionId}`}
            sx={{ pl: 4 }}
            disabled
          >
            {ver.versionLabel}
          </MenuItem>,
          ...ver.tests.map(test => (
            <MenuItem 
              key={`test-${func.functionId}-${ver.versionId}-${test.id}`}
              value={`${func.functionId}:${ver.versionId}:${test.id}`}
              sx={{ pl: 6 }}
            >
              Тест {test.date}
            </MenuItem>
          ))
        ])
      ]).flat()
    ];
  };

  // Рендеринг списка совместимых профилей для сравнения
  const renderCompatibleProfilesMenu = () => {
    if (compatibleProfiles.length === 0) {
      return [
        <MenuItem key="empty" disabled>
          Нет профилей для сравнения
        </MenuItem>
      ];
    }

    return [
      <MenuItem key="default" value="">
        Выберите профиль
      </MenuItem>,
      ...compatibleProfiles.map(profile => (
        <MenuItem 
          key={profile.id}
          value={profile.id}
        >
          {profile.versionLabel} - Тест {profile.testDate}
        </MenuItem>
      ))
    ];
  };

  // Расчет процентной разницы между метриками
  const calculateDiff = (metric) => {
    if (!profileData1 || !profileData2 || !profileData1.metrics[metric] || !profileData2.metrics[metric]) {
      return null;
    }
    
    const p50Value1 = profileData1.metrics[metric].p50;
    const p50Value2 = profileData2.metrics[metric].p50;
    
    if (p50Value1 === 0) return null;
    
    const diff = ((p50Value2 - p50Value1) / p50Value1) * 100;
    
    // Определить, является ли изменение улучшением
    let isImprovement;
    
    if (metric === 'executionTime' || metric === 'coldStart' || metric === 'cpu' || 
        metric === 'memory' || metric === 'network') {
      // Для этих метрик уменьшение - это улучшение
      isImprovement = diff < 0;
    } else {
      // Для других метрик увеличение может быть улучшением
      isImprovement = diff > 0;
    }
    
    return {
      value: diff.toFixed(1),
      isImprovement
    };
  };

  // Получение иконки языка
  const getLanguageIcon = (language) => {
    switch (language) {
      case 'python':
        return <CodeIcon fontSize="small" color="primary" />;
      case 'nodejs':
        return <JavascriptIcon fontSize="small" color="warning" />;
      default:
        return <CodeIcon fontSize="small" />;
    }
  };

  // Проверка ошибок загрузки и пустых данных
  useEffect(() => {
    if (profiles.length === 0) {
      setError('Нет сохранённых результатов. Запустите тестирование');
    } else {
      setError(null);
    }
  }, [profiles]);

  return (
    <Container maxWidth="lg">
      <Box sx={{ my: 4 }}>
        <Typography variant="h4" component="h1" gutterBottom>
          Результаты тестирования
        </Typography>

        <Grid container spacing={2} sx={{ mb: 4 }}>
          <Grid item xs={12} md={5}>
            <FormControl fullWidth>
              <InputLabel>Профиль 1</InputLabel>
              <Select
                value={selectedProfile1}
                label="Профиль 1"
                onChange={handleProfile1Change}
                MenuProps={{ style: { maxHeight: 500 } }}
              >
                {renderProfilesMenu()}
              </Select>
            </FormControl>
          </Grid>
          
          <Grid item xs={12} md={2} sx={{ display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
            <Typography variant="body1" sx={{ color: 'text.secondary' }}>
              сравнить с
            </Typography>
          </Grid>
          
          <Grid item xs={12} md={5}>
            <FormControl fullWidth disabled={!selectedProfile1}>
              <InputLabel>Профиль 2</InputLabel>
              <Select
                value={selectedProfile2}
                label="Профиль 2"
                onChange={handleProfile2Change}
                MenuProps={{ style: { maxHeight: 500 } }}
              >
                {renderCompatibleProfilesMenu()}
              </Select>
            </FormControl>
          </Grid>
        </Grid>

        {pageState === PAGE_STATES.SELECTION && (
          <Box sx={{ textAlign: 'center', py: 8 }}>
            {error ? (
              <Alert severity="info" sx={{ display: 'inline-flex' }}>
                <AlertTitle>Информация</AlertTitle>
                {error}
              </Alert>
            ) : (
              <Typography variant="h6" color="text.secondary">
                Выберите профиль для просмотра результатов тестирования
              </Typography>
            )}
          </Box>
        )}

        {(pageState === PAGE_STATES.VIEW || pageState === PAGE_STATES.COMPARISON) && (
          <>
            {loading ? (
              <Box sx={{ display: 'flex', justifyContent: 'center', py: 8 }}>
                <CircularProgress />
              </Box>
            ) : error ? (
              <Alert 
                severity="error" 
                action={
                  <Button color="inherit" size="small" onClick={() => loadProfileData(selectedProfile1, setProfileData1)}>
                    Повторить
                  </Button>
                }
              >
                <AlertTitle>Ошибка</AlertTitle>
                {error}
              </Alert>
            ) : profileData1 && (
              <>
                <Paper sx={{ mb: 4 }}>
                  <Tabs
                    value={currentTabIndex}
                    onChange={handleTabChange}
                    sx={{ borderBottom: 1, borderColor: 'divider' }}
                    variant="scrollable"
                    scrollButtons="auto"
                  >
                    {METRICS.map((metric, index) => {
                      const hasData = Boolean(profileData1.metrics[metric.id]);
                      return (
                        <Tab 
                          key={metric.id} 
                          label={metric.label} 
                          disabled={!hasData}
                          sx={{ opacity: hasData ? 1 : 0.5 }}
                        />
                      );
                    })}
                  </Tabs>
                  <Box sx={{ p: 3 }}>
                    {METRICS.map((metric, index) => {
                      if (currentTabIndex !== index) return null;
                      
                      const chartData = getMetricChartData(metric.id);
                      if (!chartData) {
                        return (
                          <Box key={metric.id} sx={{ textAlign: 'center', py: 4 }}>
                            <Typography color="text.secondary">
                              Нет данных для этой метрики
                            </Typography>
                          </Box>
                        );
                      }
                      
                      return (
                        <Box key={metric.id}>
                          <Box sx={{ height: 400 }}>
                            <Line 
                              options={getChartOptions(`${metric.label} (перцентили)`)} 
                              data={chartData} 
                            />
                          </Box>
                        </Box>
                      );
                    })}
                  </Box>
                </Paper>

                <Grid container spacing={3}>
                  {pageState === PAGE_STATES.COMPARISON ? (
                    <>
                      <Grid item xs={12} md={4} sx={{ display: 'flex' }}>
                        <Card sx={{ width: '100%', height: '100%' }}>
                          <CardContent sx={{ display: 'flex', flexDirection: 'column', height: '100%' }}>
                            <Typography variant="h6" gutterBottom>
                              Детали теста 1
                            </Typography>
                            <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                              {getLanguageIcon(profileData1.language)}
                              <Box sx={{ ml: 1 }}>
                                <Typography variant="body2">
                                  <strong>Функция:</strong> {profileData1.functionName}
                                </Typography>
                              </Box>
                            </Box>
                            <Typography variant="body2">
                              <strong>Версия:</strong> {profileData1.versionLabel}
                            </Typography>
                            <Typography variant="body2">
                              <strong>Платформа:</strong> {profileData1.platform}
                            </Typography>
                            <Typography variant="body2">
                              <strong>Рантайм:</strong> {profileData1.runtime}
                            </Typography>
                            <Typography variant="body2">
                              <strong>Дата запуска:</strong> {profileData1.date}
                            </Typography>
                            <Typography variant="body2">
                              <strong>Кол-во итераций:</strong> {profileData1.iterations}
                            </Typography>
                            <Typography variant="body2">
                              <strong>Параллельных запусков:</strong> {profileData1.concurrency}
                            </Typography>
                          </CardContent>
                        </Card>
                      </Grid>
                      
                      <Grid item xs={12} md={4} sx={{ display: 'flex' }}>
                        <Card sx={{ width: '100%', height: '100%' }}>
                          <CardContent sx={{ display: 'flex', flexDirection: 'column', height: '100%' }}>
                            <Typography variant="h6" gutterBottom>
                              Детали теста 2
                            </Typography>
                            {profileData2 && (
                              <>
                                <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                                  {getLanguageIcon(profileData2.language)}
                                  <Box sx={{ ml: 1 }}>
                                    <Typography variant="body2">
                                      <strong>Функция:</strong> {profileData2.functionName}
                                    </Typography>
                                  </Box>
                                </Box>
                                <Typography variant="body2">
                                  <strong>Версия:</strong> {profileData2.versionLabel}
                                </Typography>
                                <Typography variant="body2">
                                  <strong>Платформа:</strong> {profileData2.platform}
                                </Typography>
                                <Typography variant="body2">
                                  <strong>Рантайм:</strong> {profileData2.runtime}
                                </Typography>
                                <Typography variant="body2">
                                  <strong>Дата запуска:</strong> {profileData2.date}
                                </Typography>
                                <Typography variant="body2">
                                  <strong>Кол-во итераций:</strong> {profileData2.iterations}
                                </Typography>
                                <Typography variant="body2">
                                  <strong>Параллельных запусков:</strong> {profileData2.concurrency}
                                </Typography>
                              </>
                            )}
                          </CardContent>
                        </Card>
                      </Grid>
                      
                      <Grid item xs={12} md={4} sx={{ display: 'flex' }}>
                        <Card sx={{ width: '100%', height: '100%' }}>
                          <CardContent sx={{ display: 'flex', flexDirection: 'column', height: '100%' }}>
                            <Typography variant="h6" gutterBottom>
                              Сводка изменений
                            </Typography>
                            {profileData1 && profileData2 && (
                              <>
                                {METRICS.map(metric => {
                                  const diff = calculateDiff(metric.id);
                                  if (!diff) return null;
                                  
                                  const color = diff.isImprovement ? 'success.main' : 'error.main';
                                  const prefix = diff.value.startsWith('-') ? '' : '+';
                                  
                                  return (
                                    <Box key={metric.id} sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                      <Typography variant="body2">{metric.label}:</Typography>
                                      <Typography variant="body2" color={color}>
                                        {prefix}{diff.value}%
                                      </Typography>
                                    </Box>
                                  );
                                })}
                                <Box sx={{ flexGrow: 1 }} />
                              </>
                            )}
                          </CardContent>
                        </Card>
                      </Grid>
                    </>
                  ) : (
                    <Grid item xs={12} sx={{ display: 'flex' }}>
                      <Card sx={{ width: '100%' }}>
                        <CardContent>
                          <Typography variant="h6" gutterBottom>
                            Детали теста
                          </Typography>
                          <Grid container spacing={2}>
                            <Grid item xs={12} md={6}>
                              <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                                {getLanguageIcon(profileData1.language)}
                                <Box sx={{ ml: 1 }}>
                                  <Typography variant="body2">
                                    <strong>Функция:</strong> {profileData1.functionName}
                                  </Typography>
                                </Box>
                              </Box>
                              <Typography variant="body2">
                                <strong>Версия:</strong> {profileData1.versionLabel}
                              </Typography>
                              <Typography variant="body2">
                                <strong>Платформа:</strong> {profileData1.platform}
                              </Typography>
                              <Typography variant="body2">
                                <strong>Рантайм:</strong> {profileData1.runtime}
                              </Typography>
                            </Grid>
                            <Grid item xs={12} md={6}>
                              <Typography variant="body2">
                                <strong>Дата запуска:</strong> {profileData1.date}
                              </Typography>
                              <Typography variant="body2">
                                <strong>Кол-во итераций:</strong> {profileData1.iterations}
                              </Typography>
                              <Typography variant="body2">
                                <strong>Параллельных запусков:</strong> {profileData1.concurrency}
                              </Typography>
                            </Grid>
                          </Grid>
                        </CardContent>
                      </Card>
                    </Grid>
                  )}
                </Grid>
              </>
            )}
          </>
        )}
      </Box>
    </Container>
  );
}

export default ResultsPage; 