import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import {
  Box,
  Button,
  Container,
  FormControl,
  InputLabel,
  MenuItem,
  Paper,
  Select,
  TextField,
  Typography,
  Alert,
} from '@mui/material';
import CodeEditor from '@uiw/react-textarea-code-editor';

// Моковые данные для функций
const mockFunctions = {
  'func1': {
    name: 'API Gateway Handler',
    language: 'python',
    platform: 'yandex',
    versions: {
      1: {
        code: `import json

def handler(event, context):
    """
    Обработчик запросов API Gateway
    """
    request_body = json.loads(event['body'])
    user_id = request_body.get('user_id')
    action = request_body.get('action')
    
    # Проверка параметров
    if not user_id or not action:
        return {
            'statusCode': 400,
            'body': json.dumps({
                'error': 'Missing required parameters'
            })
        }
    
    # Обработка запроса
    result = process_request(user_id, action)
    
    # Возврат результата
    return {
        'statusCode': 200,
        'body': json.dumps(result)
    }

def process_request(user_id, action):
    # Здесь могла бы быть бизнес-логика
    return {
        'user_id': user_id,
        'action': action,
        'status': 'processed'
    }`,
        type: 'manual'
      },
      2: {
        code: `import json

def handler(event, context):
    """
    Оптимизированный обработчик запросов API Gateway
    """
    request_body = json.loads(event['body'])
    user_id = request_body.get('user_id')
    action = request_body.get('action')
    
    # Быстрая проверка параметров
    if not user_id or not action:
        return {
            'statusCode': 400,
            'body': '{"error":"Missing required parameters"}'
        }
    
    # Обработка запроса
    result = process_request(user_id, action)
    
    # Возврат результата с предварительным форматированием JSON
    return {
        'statusCode': 200,
        'body': json.dumps({
            'user_id': user_id,
            'action': action,
            'status': 'processed'
        })
    }

def process_request(user_id, action):
    # Оптимизированная бизнес-логика
    return {
        'user_id': user_id,
        'action': action,
        'status': 'processed'
    }`,
        type: 'auto',
        optimizationDetails: [
          'Уменьшено количество вызовов json.dumps для повышения производительности',
          'Предварительное форматирование JSON в случае ошибки для экономии ресурсов',
          'Оптимизирована структура возвращаемых данных для снижения задержек'
        ]
      },
      3: {
        code: `import json
import time

def handler(event, context):
    """
    Улучшенный обработчик запросов API Gateway с метриками
    """
    start_time = time.time()
    
    request_body = json.loads(event['body'])
    user_id = request_body.get('user_id')
    action = request_body.get('action')
    
    # Быстрая проверка параметров
    if not user_id or not action:
        return {
            'statusCode': 400,
            'body': '{"error":"Missing required parameters"}'
        }
    
    # Обработка запроса
    result = process_request(user_id, action)
    
    # Добавление метрик производительности
    execution_time = time.time() - start_time
    
    # Возврат результата с предварительным форматированием JSON
    return {
        'statusCode': 200,
        'body': json.dumps({
            'user_id': user_id,
            'action': action,
            'status': 'processed',
            'execution_time_ms': round(execution_time * 1000, 2)
        })
    }

def process_request(user_id, action):
    # Оптимизированная бизнес-логика
    return {
        'user_id': user_id,
        'action': action,
        'status': 'processed'
    }`,
        type: 'manual'
      }
    }
  },
  'func2': {
    name: 'Database Worker',
    language: 'nodejs',
    platform: 'yandex',
    versions: {
      1: {
        code: `const { Client } = require('pg');

exports.handler = async function(event, context) {
  // Создаем подключение к БД
  const client = new Client({
    user: process.env.DB_USER,
    host: process.env.DB_HOST,
    database: process.env.DB_NAME,
    password: process.env.DB_PASSWORD,
    port: process.env.DB_PORT,
  });
  
  // Подключаемся к БД
  await client.connect();
  
  try {
    // Получаем параметры из запроса
    const requestData = JSON.parse(event.body);
    const { query, params } = requestData;
    
    // Выполняем запрос
    const result = await client.query(query, params);
    
    // Возвращаем результат
    return {
      statusCode: 200,
      body: JSON.stringify(result.rows)
    };
  } catch (err) {
    // Обработка ошибок
    return {
      statusCode: 500,
      body: JSON.stringify({ error: err.message })
    };
  } finally {
    // Закрываем соединение
    await client.end();
  }
};`,
        type: 'manual'
      },
      2: {
        code: `const { Pool } = require('pg');

// Создаем пул соединений для повторного использования
const pool = new Pool({
  user: process.env.DB_USER,
  host: process.env.DB_HOST,
  database: process.env.DB_NAME,
  password: process.env.DB_PASSWORD,
  port: process.env.DB_PORT,
  max: 20, // максимальное количество соединений
  idleTimeoutMillis: 30000, // время ожидания перед закрытием неиспользуемых соединений
});

exports.handler = async function(event, context) {
  // Получаем клиента из пула
  const client = await pool.connect();
  
  try {
    // Получаем параметры из запроса
    const requestData = JSON.parse(event.body);
    const { query, params } = requestData;
    
    // Выполняем запрос
    const result = await client.query(query, params);
    
    // Возвращаем результат
    return {
      statusCode: 200,
      body: JSON.stringify(result.rows)
    };
  } catch (err) {
    // Обработка ошибок
    return {
      statusCode: 500,
      body: JSON.stringify({ error: err.message })
    };
  } finally {
    // Возвращаем клиента в пул
    client.release();
  }
};`,
        type: 'auto',
        optimizationDetails: [
          'Заменен Client на Pool для повторного использования соединений БД',
          'Добавлены настройки пула для оптимального управления ресурсами',
          'Использован метод client.release() вместо client.end() для возврата соединения в пул'
        ]
      }
    }
  },
  'func3': {
    name: 'Image Processor',
    language: 'python',
    platform: 'yandex',
    versions: {
      1: {
        code: `import io
import json
import base64
from PIL import Image

def handler(event, context):
    """
    Обработчик изображений из запроса
    """
    try:
        # Декодируем изображение из base64
        image_data = json.loads(event['body']).get('image')
        if not image_data:
            return {
                'statusCode': 400,
                'body': json.dumps({
                    'error': 'No image data provided'
                })
            }
        
        # Декодируем base64 в байты
        image_bytes = base64.b64decode(image_data)
        
        # Открываем изображение с помощью PIL
        image = Image.open(io.BytesIO(image_bytes))
        
        # Обрабатываем изображение (пример: изменение размера)
        processed_image = image.resize((300, 300))
        
        # Сохраняем обработанное изображение в буфер
        buffer = io.BytesIO()
        processed_image.save(buffer, format='JPEG')
        buffer.seek(0)
        
        # Кодируем обратно в base64
        processed_image_base64 = base64.b64encode(buffer.getvalue()).decode('utf-8')
        
        return {
            'statusCode': 200,
            'body': json.dumps({
                'processed_image': processed_image_base64
            })
        }
    except Exception as e:
        return {
            'statusCode': 500,
            'body': json.dumps({
                'error': str(e)
            })
        }`,
        type: 'manual'
      }
    }
  }
};

// Новый шаблон для создания функции
const emptyTemplates = {
  'python': `# Введите свой код здесь
def handler(event, context):
    """
    Функция-обработчик для Yandex Cloud Functions
    """
    return {
        'statusCode': 200,
        'body': '{"message": "Hello from serverless function!"}'
    }`,
  'nodejs': `// Введите свой код здесь
exports.handler = async function(event, context) {
  return {
    statusCode: 200,
    body: JSON.stringify({ message: "Hello from serverless function!" })
  };
};`
};

// Enum для состояний страницы
const PageState = {
  CREATE: 'create',
  VIEW: 'view',
  VIEW_OPTIMIZED: 'view_optimized',
  EDIT: 'edit'
};

function OptimizationPage() {
  const { functionId, version } = useParams();
  const navigate = useNavigate();
  
  // Состояния для различных полей
  const [pageState, setPageState] = useState(PageState.CREATE);
  const [functionName, setFunctionName] = useState('');
  const [language, setLanguage] = useState('python');
  const [platform, setPlatform] = useState('yandex');
  const [code, setCode] = useState(emptyTemplates.python);
  const [currentVersion, setCurrentVersion] = useState(1);
  const [optimizationDetails, setOptimizationDetails] = useState([]);
  const [alertVisible, setAlertVisible] = useState(false);
  
  // Ref для синхронизации прокрутки
  const codeEditorRef = React.useRef(null);
  const lineNumbersRef = React.useRef(null);
  
  // Функция для синхронизации прокрутки
  const handleScroll = React.useCallback((e) => {
    if (lineNumbersRef.current) {
      lineNumbersRef.current.scrollTop = e.target.scrollTop;
    }
  }, []);
  
  // Обработка изменения состояния страницы на основе URL параметров
  useEffect(() => {
    if (functionId && version) {
      // Получаем данные функции из моковых данных
      const funcData = mockFunctions[functionId];
      if (funcData && funcData.versions[version]) {
        setFunctionName(funcData.name);
        setLanguage(funcData.language);
        setPlatform(funcData.platform);
        setCode(funcData.versions[version].code);
        setCurrentVersion(parseInt(version));
        
        // Устанавливаем состояние на основе типа версии
        if (funcData.versions[version].type === 'auto') {
          setPageState(PageState.VIEW_OPTIMIZED);
          setOptimizationDetails(funcData.versions[version].optimizationDetails || []);
        } else {
          setPageState(PageState.VIEW);
        }
      }
    } else if (functionId && !version) {
      // Если есть ID функции, но нет версии - открываем последнюю версию
      const funcData = mockFunctions[functionId];
      if (funcData) {
        const latestVersion = Math.max(...Object.keys(funcData.versions).map(v => parseInt(v)));
        navigate(`/optimization/${functionId}/${latestVersion}`);
      }
    } else {
      // Если нет параметров - режим создания новой функции
      setPageState(PageState.CREATE);
      setFunctionName('');
      setLanguage('python');
      setPlatform('yandex');
      setCode(emptyTemplates.python);
      setCurrentVersion(1);
      setOptimizationDetails([]);
    }
  }, [functionId, version, navigate]);
  
  // Обновление шаблона кода при смене языка в режиме создания
  useEffect(() => {
    if (pageState === PageState.CREATE) {
      setCode(emptyTemplates[language]);
    }
  }, [language, pageState]);
  
  // Обработчики событий
  const handleSaveFunction = () => {
    // Здесь должен быть запрос к API для сохранения функции
    // В данном прототипе просто показываем уведомление и меняем состояние
    setAlertVisible(true);
    setTimeout(() => setAlertVisible(false), 3000);
    
    if (pageState === PageState.CREATE) {
      // После создания переходим к просмотру
      setPageState(PageState.VIEW);
    } else if (pageState === PageState.EDIT) {
      // После редактирования переходим к просмотру с новой версией
      setCurrentVersion(currentVersion + 1);
      setPageState(PageState.VIEW);
    }
  };
  
  const handleOptimizeCode = () => {
    // Здесь должен быть запрос к API для оптимизации кода через LLM
    // В данном прототипе просто увеличиваем версию и добавляем фиктивные детали оптимизации
    setCurrentVersion(currentVersion + 1);
    setOptimizationDetails([
      'Оптимизирована структура кода для улучшения читаемости',
      'Улучшена обработка ошибок для повышения надежности',
      'Добавлено кэширование для улучшения времени ответа'
    ]);
    setPageState(PageState.VIEW_OPTIMIZED);
  };
  
  const handleEditCode = () => {
    setPageState(PageState.EDIT);
  };
  
  // eslint-disable-next-line no-unused-vars
  const handleCreateNewFunction = () => {
    navigate('/optimization');
  };
  
  // Определяем, доступна ли кнопка сохранения
  const isSaveButtonDisabled = () => {
    if (pageState === PageState.CREATE) {
      return !functionName || !code.trim();
    } else if (pageState === PageState.EDIT) {
      // Для Edit состояния нужно проверить, был ли изменен код
      // В реальном приложении нужно сравнивать с оригинальным кодом
      return !code.trim();
    }
    return true;
  };
  
  return (
    <Container maxWidth="lg">
      <Box sx={{ my: 4 }}>
        {/* Заголовок и версия */}
        <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }}>
          {pageState === PageState.CREATE ? (
            <TextField
              label="Название"
              variant="outlined"
              fullWidth
              value={functionName}
              onChange={(e) => setFunctionName(e.target.value)}
              placeholder="Новая функция"
              sx={{ fontSize: '1.5rem' }}
            />
          ) : (
            <Typography variant="h4" component="h1" gutterBottom>
              {functionName}
            </Typography>
          )}
          
          {pageState !== PageState.CREATE && (
            <Box sx={{ ml: 3, display: 'flex', alignItems: 'center' }}>
              <Typography variant="subtitle1" component="span" fontWeight="medium">
                Версия:
              </Typography>
              <Typography variant="subtitle1" component="span" sx={{ ml: 1 }}>
                {currentVersion}
              </Typography>
            </Box>
          )}
        </Box>
        
        {/* Селекторы для языка и платформы (только в режиме создания) */}
        {pageState === PageState.CREATE && (
          <Box sx={{ display: 'flex', gap: 2, mb: 3 }}>
            <FormControl sx={{ minWidth: 200 }}>
              <InputLabel>Язык</InputLabel>
              <Select
                value={language}
                label="Язык"
                onChange={(e) => setLanguage(e.target.value)}
              >
                <MenuItem value="python">Python</MenuItem>
                <MenuItem value="nodejs">Node.js</MenuItem>
              </Select>
            </FormControl>
            
            <FormControl sx={{ minWidth: 200 }}>
              <InputLabel>Облачная платформа</InputLabel>
              <Select
                value={platform}
                label="Облачная платформа"
                onChange={(e) => setPlatform(e.target.value)}
              >
                <MenuItem value="yandex">Yandex Cloud Functions</MenuItem>
              </Select>
            </FormControl>
          </Box>
        )}
        
        {/* Редактор кода */}
        <Paper 
          elevation={3} 
          sx={{ 
            p: 0, 
            mb: 3, 
            borderRadius: 1, 
            overflow: 'hidden',
            height: '60vh',
            display: 'flex',
            flexDirection: 'column',
            position: 'relative'
          }}
        >
          <Box
            ref={lineNumbersRef}
            className="line-numbers"
            sx={{
              position: 'absolute',
              top: 0,
              left: 0,
              height: '100%',
              width: '42px',
              backgroundColor: '#f0f0f0',
              borderRight: '1px solid #ddd',
              overflowY: 'hidden',
              zIndex: 1,
              pt: '15px',
              pb: '15px',
              userSelect: 'none',
              color: '#666',
              textAlign: 'right',
              fontSize: '14px',
              fontFamily: '"JetBrains Mono", monospace',
              lineHeight: 1.5
            }}
          >
            {code.split('\n').map((_, i) => (
              <div key={i} style={{ paddingRight: '8px' }}>{i + 1}</div>
            ))}
          </Box>
          <CodeEditor
            ref={codeEditorRef}
            value={code}
            language={language === 'python' ? 'python' : 'js'}
            onChange={(e) => {
              if (pageState === PageState.CREATE || pageState === PageState.EDIT) {
                setCode(e.target.value);
              }
            }}
            onScroll={handleScroll}
            padding={15}
            data-color-mode="light"
            style={{
              fontSize: '14px',
              fontFamily: '"JetBrains Mono", monospace',
              flex: 1,
              overflowY: 'auto',
              backgroundColor: '#f5f5f5',
              minHeight: '100%',
              resize: 'none',
              border: 'none',
              outline: 'none',
              paddingLeft: '50px',
              // Блокировка редактирования в режимах просмотра
              readOnly: pageState === PageState.VIEW || pageState === PageState.VIEW_OPTIMIZED,
            }}
          />
        </Paper>
        
        {/* Блок с описанием оптимизаций (только для VIEW_OPTIMIZED) */}
        {pageState === PageState.VIEW_OPTIMIZED && (
          <Paper elevation={2} sx={{ p: 3, mb: 3 }}>
            <Typography variant="h6" gutterBottom>
              Оптимизации
            </Typography>
            <ul>
              {optimizationDetails.map((detail, index) => (
                <li key={index}>
                  <Typography variant="body1">{detail}</Typography>
                </li>
              ))}
            </ul>
          </Paper>
        )}
        
        {/* Кнопки действий в зависимости от состояния */}
        <Box sx={{ display: 'flex', justifyContent: 'flex-end', gap: 2 }}>
          {/* Кнопки для CREATE состояния */}
          {pageState === PageState.CREATE && (
            <Button
              variant="contained"
              color="primary"
              onClick={handleSaveFunction}
              disabled={isSaveButtonDisabled()}
            >
              Сохранить
            </Button>
          )}
          
          {/* Кнопки для VIEW состояния */}
          {pageState === PageState.VIEW && (
            <>
              <Button
                variant="outlined"
                onClick={handleEditCode}
              >
                Редактировать
              </Button>
              <Button
                variant="contained"
                color="primary"
                onClick={handleOptimizeCode}
              >
                Оптимизировать
              </Button>
            </>
          )}
          
          {/* Кнопки для VIEW_OPTIMIZED состояния */}
          {pageState === PageState.VIEW_OPTIMIZED && (
            <Button
              variant="outlined"
              onClick={handleEditCode}
            >
              Редактировать
            </Button>
          )}
          
          {/* Кнопки для EDIT состояния */}
          {pageState === PageState.EDIT && (
            <Button
              variant="contained"
              color="primary"
              onClick={handleSaveFunction}
              disabled={isSaveButtonDisabled()}
            >
              Сохранить
            </Button>
          )}
        </Box>
        
        {/* Предупреждение об успешном сохранении */}
        {alertVisible && (
          <Alert 
            severity="success" 
            sx={{ 
              position: 'fixed', 
              bottom: 16, 
              right: 16, 
              zIndex: 9999 
            }}
          >
            Функция успешно сохранена
          </Alert>
        )}
      </Box>
    </Container>
  );
}

export default OptimizationPage; 