import math
import unittest
import numpy


def convert_precision(tolerance):
    """
    Конвертирует tolerance в порядок точности.

    Args:
        tolerance (float): Значение tolerance.

    Returns:
        int: Порядок точности.

    Raises:
        ValueError: При невозможности определить порядок точности.
    """
    if tolerance == 0:
        return 0
    return abs(math.floor(math.log10(abs(tolerance))))

def calculate(action, *args, tolerance=1e-6):

    """
    Производит действие action над аргументами *args.

    Args:
        action (string): Дейсвие, может быть одним из
             +, -, *, /, medium, variance, std_deviation, median, q2, q3 
        *args (float): Числа над которыми будет произведено действие actoin
        tolerance (float): Точность вычислений (по умолчанию 1e-6).

    Returns:
        float: Результат вычисления с округлением до указанной точности.

    Raises:
        ValueError: При неверном значении операции.
    """

    if action not in ['+', '-', '*', '/', 'medium', \
            'variance', 'std_deviation', 'median', 'q2', 'q3']:
        raise ValueError("Неверная операция")
    
    if (action == '+'):
        result = sum(args)
    elif (action == '-'):
        result = args[0]
        for i in range(1, len(args)):
           result -= args[i]
    elif (action == '*'):
        result = 1
        for i in range(len(args)):
           result *= args[i]
    elif (action == '/'):
        result = args[0]
        for i in range(1, len(args)):
           result /= args[i]

    elif (action == 'medium'):
        return numpy.mean(args)
    elif (action == 'variance'):
        return numpy.var(args)
    elif (action == 'std_deviation'):
        return numpy.std(args)
    elif (action == 'median'):
        return numpy.median(args)
    elif (action == 'q2'):
        return numpy.median(args)
    elif (action == 'q3'):
        return numpy.percentile(args, 75)
    else:
        raise ValueError("Неверная операция")
    # Округление результата до нужной точности
    precision = convert_precision(tolerance)
    rounded_result = round(result, precision)
    
    return rounded_result

class TestCalculator(unittest.TestCase):

    def test_convert_precision(self):
        self.assertEqual(convert_precision(0), 0)
        self.assertEqual(convert_precision(1e-9), 9)
        self.assertEqual(convert_precision(1e-5), 5)

    def test_calculate_addition(self):
        self.assertAlmostEqual(calculate('+', 1, 2), 3, places=8)

    def test_calculate_subtraction(self):
        self.assertAlmostEqual(calculate('-', 5, 3), 2, places=8)

    def test_calculate_multiplication(self):
        self.assertAlmostEqual(calculate('*', 4, 5), 20, places=8)

    def test_calculate_division(self):
        self.assertAlmostEqual(calculate('/', 10, 2), 5, places=8)

    def test_calculate_with_custom_tolerance(self):
        self.assertAlmostEqual(calculate('+', 0.1, 0.2), 0.3, places=8)

    def test_stat_operations(self):
        self.assertAlmostEqual(calculate('medium', 1, 2, 3, 4), 2.5, places=8)
        self.assertAlmostEqual(calculate('variance', 1, 2, 3, 4), 1.25, places=8)
        self.assertAlmostEqual(calculate('std_deviation', 1, 2, 3, 4), 0.7071067811865476, places=8)
        self.assertAlmostEqual(calculate('median', 1, 2, 3, 4), 2, places=8)
        self.assertAlmostEqual(calculate('q2', 1, 2, 3, 4), 2, places=8)
        self.assertAlmostEqual(calculate('q3', 1, 2, 3, 4), 3, places=8)

if __name__ == '__main__':
    unittest.main()
