import math
import unittest

def convert_precision(tolerance):
    """
    Конвертирует tolerance в порядок точности.

    Args:
        tolerance (float): Значение tolerance.

    Returns:
        int: Порядок точности.

    Raises:
        ValueError: При невозможности определить порядок точности.
    """
    if tolerance == 0:
        return 0
    return abs(math.floor(math.log10(abs(tolerance))))

def calculate(x, op, y, tolerance=1e-6):
    """
    Считает x op y, где x и y - числа любого типа, а op - строка операции.

    Args:
        x (float): Первое число.
        op (str): Операция (+, -, *, /).
        y (float): Второе число.
        tolerance (float): Точность вычислений (по умолчанию 1e-6).

    Returns:
        float: Результат вычисления с округлением до указанной точности.

    Raises:
        ValueError: При неверном значении операции.
    """
    if op not in ['+', '-', '*', '/']:
        raise ValueError("Неверная операция")
    
    if (op == "+"):
        result =  x + y
    elif (op == "-"):
        result =  x - y
    elif (op == "*"):
        result =  x * y
    elif (op == "/"):
        result =  x / y
    
    # Округление результата до нужной точности
    precision = convert_precision(tolerance)
    rounded_result = round(result, precision)
    
    return rounded_result

class TestCalculator(unittest.TestCase):

    def test_convert_precision(self):
        self.assertEqual(convert_precision(0), 0)
        self.assertEqual(convert_precision(1e-9), 9)
        self.assertEqual(convert_precision(1e-5), 5)


    def test_calculate_addition(self):
        self.assertAlmostEqual(calculate(1, '+', 2), 3, places=8)

    def test_calculate_subtraction(self):
        self.assertAlmostEqual(calculate(5, '-', 3), 2, places=8)

    def test_calculate_multiplication(self):
        self.assertAlmostEqual(calculate(4, '*', 5), 20, places=8)

    def test_calculate_division(self):
        self.assertAlmostEqual(calculate(10, '/', 2), 5, places=8)

    def test_calculate_with_custom_tolerance(self):
        self.assertAlmostEqual(calculate(0.1, '+', 0.2), 0.3, places=8)

if __name__ == '__main__':
    unittest.main()
